'*****************************************************************************
'*  Name    : RaceCarWithSD.bas                                              *
'*  Author  : Gavin Wiggett                                                  *
'*  Notice  : Copyright (c) 2013 Bitfogav                                    *
'*          : All Rights Reserved                                            *
'*  Date    : 18/06/2013                                                     *
'*  Version : 1.0                                                            *
'*  Notes   : Race Car Code with SD Card (Track) for Game Shield             *
'*          : Load track from SD Card "track.txt" if err with SD Card then   *
'*          : default track will be loaded.                                  *
'*****************************************************************************

' Imports section...
imports GameShield   ' Game Shield 
imports SD           ' SD Card


' Track data buffer...
dim TrackData(80) as byte              ' track buffer

' Game buffers...
dim VidRam(8) as ushort                ' video ram data buffer
dim TempVidRam(8) as ushort            ' video ram temp buffer

' Game Variables...
dim GameMode as boolean = false        ' true = in game mode, false = not in game mode
dim TrackDelay as byte = 0             ' track delay counter    
dim TrackSpeed as byte = 28            ' this sets the starting track speed
dim CountTrackLen as byte = 0          ' track counter

' Car and delay starting point...
dim RaceCarPos as byte = &B00010000    ' data for the race car
dim KeyDebounce as byte = 15           ' key debounce counter

' Level of game...
dim Level as byte = &B10000000         ' set level (1)
dim WhichLevel as byte = 0             ' level counter

' SD card filename and data byte...
dim FileName as string = "track.txt"

' Main prog starting point...
Sub Main() 
   If SD.Init <> errOK Then         ' initialise the SD card.
      InitRaceTrackDefault()        ' Load default track
      FlashLedOnDisplay(0)          ' show Load SD Failed
      'Console.Write("Failed to initialise",13,10)      
   elseif SD.Open(Filename) <> errOK then
      InitRaceTrackDefault()        ' Load default track
      FlashLedOnDisplay(1)          ' show Open SD Failed
      'Console.Write("Failed to open ", Filename, 13,10)
   else                  
      GetTrackDataSDcard()          ' Load track from SD Card
      FlashLedOnDisplay(2)          ' show successful SD read
      'Console.Write(13,10,"Finished Reading SD Card",13,10,13,10)
   end if  
   
   ' Main prog loop...
   While True      
      DrawTrack()
      Delays()
      GameShield.UpdateMatrix() 
      UpdateCar()
      CrashDetection()
   End While
End Sub

' Draw track on display...
sub DrawTrack()
   if TrackDelay = 0 and GameMode = true then         
      if CountTrackLen = sizeof(TrackData) then
         CountTrackLen = 0
         TrackSpeed = TrackSpeed - 5         ' Increase track speed 
         UpdateLevel()                       ' update level indicator
         if TrackSpeed < 9 then
            TrackSpeed = 9                   ' set max speed limit of track 
         end if
      end if                       
      ' copy old track to new position (this makes the track scroll)
      for LoadRam as byte = 1 to 7
      VidRam(LoadRam) = TempVidRam(LoadRam - 1)     
      next                                            
      VidRam(0) = TrackData(CountTrackLen)                        ' get new track data                   
      GameShield.MatrixData.GreenData(0) = VidRam(7)              ' update display data
      GameShield.MatrixData.GreenData(1) = VidRam(6)              ' "
      GameShield.MatrixData.GreenData(2) = VidRam(5)              ' "
      GameShield.MatrixData.GreenData(3) = VidRam(4)              ' "
      GameShield.MatrixData.GreenData(4) = VidRam(3)              ' "
      GameShield.MatrixData.GreenData(5) = VidRam(2)
      GameShield.MatrixData.GreenData(6) = VidRam(1)
      GameShield.MatrixData.GreenData(7) = VidRam(0)                                
      for currentRam as byte = 0 to 6                  ' temp store current video ram
         TempVidRam(CurrentRam) = VidRam(CurrentRam)       
      next             
      TrackDelay = TrackSpeed                          ' set track delay
      CountTrackLen += 1                               ' increase our track counter      
   end if
end sub

' Track & Button delays...
sub delays()                                ' check delays and decrease
   if TrackDelay <> 0 then                  ' track delay counter
      TrackDelay -= 1                       '
   end if                                   ' 
   if KeyDebounce <> 0 then                 ' key debounce counter
      KeyDebounce -= 1                      '
   end if
end sub

' Update car position track / start new game...
sub UpdateCar()
   if KeyDebounce = 0 then
      dim Key as byte = Keypress()
         select Key
            case 1 ' Select key (start game)
               if GameMode = false then                
                  GameShield.ClearMatrix()                                            
                  erase(VidRam)                                   ' clear buffers
                  erase(TempVidRam)                               ' "
                  erase(GameShield.MatrixData.GreenData)          ' "                    
                  CountTrackLen = 0                               ' reset track counter                                 
                  GameShield.ClearRedData()                       ' reset level
                  WhichLevel = 0                                  ' "
                  RaceCarPos = &B00010000                         ' set racecar posisiton
                  GameShield.MatrixData.RedData(0) = RaceCarPos   ' update car data    
                  TrackSpeed = 28                                 ' reset track speed     
                  GameMode = true                                 ' set game mode to True                        
                  delayms(1000)                                   ' slight delay before start game 
               end if
            case 4 ' Left key
               if GameMode = true and not RaceCarPos = &B01000000 then  ' only move car if not on left edge
                  RaceCarPos = RaceCarPos << 1
                  GameShield.MatrixData.RedData(0) = RaceCarPos         '  update car data
                  KeyDebounce = 15          
               end if  
            case 5 ' Right key
               if GameMode = true and not RaceCarPos = &B00000010 then  ' only move car if not on right edge 
                  RaceCarPos = RaceCarPos >> 1
                  GameShield.MatrixData.RedData(0) = RaceCarPos         '  update car data
                  KeyDebounce = 15                                               
               end if                   
         end select
   end if
end sub

' Crash Detection (check if car has crashed)...
sub CrashDetection()
   if GameMode = true then
      dim TempCrash as byte = RaceCarPos and GameShield.MatrixData.GreenData(0) 
      if TempCrash <> 0 then
         GameMode = false
      end if          
   end if
end sub

' Update level indicator (left side of display)...
sub UpdateLevel()
   WhichLevel += 1                                          ' increase level counter
   Level = &B10000000                                       ' set level indicator
   if WhichLevel = 1 then
   GameShield.MatrixData.RedData(WhichLevel) = Level        ' set new level data position
   else
   GameShield.MatrixData.RedData(WhichLevel -1) = 0         ' clear old level data
   GameShield.MatrixData.RedData(WhichLevel) = Level        ' set new level data position   
   end if
   if whichLevel = 8 then                                   ' check to see if we have finished the race
      GameMode = false                                      ' set game mode to False
   end if
end sub  

' Get track data from SD Card (if error default track is loaded)...
sub GetTrackDataSDcard()                                     
   dim DataByte as byte = 0              ' temp buffer for sd read byte
   dim BitCount as byte = 0              ' used to set bit of byte
   dim CounterForArray as byte = 0       ' used to set Array location 
   dim DataByteCopy(80) as byte          ' temp location for track data from sd card
      do               
         DataByte = SD.ReadByte  ' read byte (this is a single character) or a ascii value.
               
         ' As we read a byte at a time from sd card as a ascii value, we need to convert it 
         ' back into a binary value 1 or 0 as only binary numbers will be used in the
         ' sd card values, any other data in the file will corrupt the data process.
         if BitCount >= 0 and BitCount <= 7 then
           if DataByte = 49 then
               DataByteCopy(counterForArray).Bits(BitCount) = 1
            elseif DataByte = 48 then ' else value is 48 = 0
               DataByteCopy(CounterForArray).Bits(BitCount) = 0
            end if                            
         end if        
         BitCount += 1                                             
         if BitCount = 10 then         ' we need to ignore the the last few bytes 
            BitCount = 0               ' (New Line (10 ascii=A)) so bitCount is set to 10
            CounterForArray += 1       ' now increase out counter for our next Array byte
         end if            
      loop Until SD.EOF              
      SD.Close()                      
   for i as byte = 0 to ubound(DataByteCopy) 
      TrackData(i) = Reverse(DataByteCopy(i),8)  ' Reverse the bytes to match track data
   next 
end sub                        

' Flash a led on Display, used to show if SD read err ok...
' Green = ok, Orange = SD Failed, Red = SD Load Failed
sub FlashLedOnDisplay(byval pType as byte)
   for x as byte = 0 to 4                             
      select pType
         case 0 : GameShield.MatrixData.RedData(7) = 1            ' Red led, Load SD Failed
         case 1
            GameShield.MatrixData.RedData(7) = 1                  ' Orange led, Open SD Failed
            GameShield.MatrixData.GreenData(7) = 1 
         case else  : GameShield.MatrixData.GreenData(7) = 1      ' Green led, Successful SD read
      end select      
      for i as byte = 0 to 30                                     ' show Led on
         UpdateMatrix()              
      next
      GameShield.ClearMatrix()                                   ' clear display (turn off led)
      delayms(400)        
   next     
   GameShield.ClearRedData()                                      ' make sure data is cleared
   GameShield.ClearGreenData()                                    ' make sure data is cleared
end sub

' Init Race Track (this is Default track data)...
inline sub InitRaceTrackDefault()
   TrackData(0)  = &B10000001      '
	TrackData(1)  = &B10000001      '
	TrackData(2)  = &B10011001      '
	TrackData(3)  = &B10011001      '
	TrackData(4)  = &B10011001      '
	TrackData(5)  = &B10000001      '
	TrackData(6)  = &B11000001      '
	TrackData(7)  = &B11000011      '
	TrackData(8)  = &B11000111      '
	TrackData(9)  = &B11100111      '
	TrackData(10) = &B11100011      '
	TrackData(11) = &B11110011      '
	TrackData(12) = &B11110001      '
	TrackData(13) = &B11110001      '
	TrackData(14) = &B11100001      '
	TrackData(15) = &B11000011      '
	TrackData(16) = &B10000111      '
	TrackData(17) = &B10001111      '
	TrackData(18) = &B10001111      '
	TrackData(19) = &B10000111      '
	TrackData(20) = &B10000011      '
	TrackData(21) = &B10100001      '
	TrackData(22) = &B10110001      '
	TrackData(23) = &B10110001      '
	TrackData(24) = &B10010001      '
	TrackData(25) = &B10000001      '
	TrackData(26) = &B10000011      '
	TrackData(27) = &B11000111      '
	TrackData(28) = &B11000111      '
	TrackData(29) = &B11000111      '
	TrackData(30) = &B10000011      '
	TrackData(31) = &B10000001      '
	TrackData(32) = &B10011001      '
	TrackData(33) = &B10011001      '
	TrackData(34) = &B10011001      '
	TrackData(35) = &B10011001      '
	TrackData(36) = &B10000001      '
	TrackData(37) = &B10000011      '
	TrackData(38) = &B11000111      '
	TrackData(39) = &B11100111      '
	TrackData(40) = &B11100011      '
	TrackData(41) = &B11000001      '
	TrackData(42) = &B10000001      '
	TrackData(43) = &B10000001      '
	TrackData(44) = &B10011001      '
	TrackData(45) = &B10011001      '
	TrackData(46) = &B10011001      '
	TrackData(47) = &B10011001      '
	TrackData(48) = &B10000001      '
	TrackData(49) = &B11000011      '
	TrackData(50) = &B11100111      '
	TrackData(51) = &B11100111      '
	TrackData(52) = &B11000111      '
	TrackData(53) = &B11000011      '
	TrackData(54) = &B11100001      '
	TrackData(55) = &B11110001      '
	TrackData(56) = &B11110001      '
	TrackData(57) = &B11110001      '
	TrackData(58) = &B11110001      '
	TrackData(59) = &B11100011      '
	TrackData(60) = &B11100111      '
	TrackData(61) = &B11000111      '
	TrackData(62) = &B11000111      '
	TrackData(63) = &B11000111      '
	TrackData(64) = &B11000111      '
	TrackData(65) = &B11100111      '
	TrackData(66) = &B11100011      '
	TrackData(67) = &B11000011      '
	TrackData(68) = &B10000011      '
	TrackData(69) = &B10000011      '
	TrackData(70) = &B10011001      '
	TrackData(71) = &B10011001      '
	TrackData(72) = &B10000011      '
	TrackData(73) = &B11000011      '
	TrackData(74) = &B11000111      '
	TrackData(75) = &B11000111      '
	TrackData(76) = &B11100011      '
	TrackData(77) = &B11100001      '
	TrackData(78) = &B11000001      ' 
   TrackData(79) = &B11000001      ' 
   erase(VidRam)                   ' clear buffers
   erase(TempVidRam) 
end sub 
